Attribute VB_Name = "ExcelInterface"
Option Explicit
'   --------------EXCEL AUTOMATION CODE--------------
'
'   ExcelSave():    Takes the data in the VB Arrays and stores them as an .xls
'                   spreadsheet document.
'
'   ExcelUpdate():  Main called routine.  Calls functions which, cell by cell,
'                   put current data into the Excel template.
'
'   ChartSetup():   Called by ExcelUpdate(), changes the chart parameters after
'                   new data is fed into the template.
'
'   AddAddIns():    Adds the Analysis Toolpak addins.
'
'   RemoveAddIns(): Removes the Analysis Toolpak addins.
'
'   FFT():          Executes the FFT algorithm within Excel on the (just) updated
'                   data.
'
'   WriteDataToCells(): Called by ExcelUpdate(), does exactly what the name says.
'
'   FrequencyUpdate():  [Conditionally] Called by ExcelUpdate(), puts the current
'                       calculated values of Gain and Phase into Excel's other
'                       worksheet.
'
'   ClearFrequency():   Called by ExcelUpdate() when the check box to reset
'                       frequency response data is true, this clears that data.
'

Public Sub ExcelSave(InputWaveData() As Single, OutputWaveData() As Single, FileName As String, NumberOfSamples As Integer)
    'Takes data stored in current VB arrays, saves them in Excel (8.0) format
    
    Dim i As Integer

    ' Declare object variables for Microsoft Excel,
   ' application workbook, and worksheet objects.
   
   Dim xlApp As Excel.Application
   Dim xlBook As Excel.Workbook
   Dim xlSheet As Excel.Worksheet
   Dim xlChart As Excel.Chart
   
   ' Assign object references to the variables. Use
   ' Add methods to create new workbook and worksheet objects.
   Set xlApp = New Excel.Application
   Set xlBook = xlApp.Workbooks.Add
   Set xlSheet = xlBook.Worksheets("Sheet1")
   
   ' Assign the values entered in the text boxes to   ' Microsoft Excel cells.
   
   With xlSheet
    For i = 0 To NumberOfSamples - 1
        .Cells(i + 2, 1) = i + 1
        .Cells(i + 2, 2).value = OutputWaveData(i)
        .Cells(i + 2, 3).value = InputWaveData(i)
    Next i
   End With
   
   
   xlSheet.SaveAs FileName                      ' Save the Worksheet.
   xlBook.Close                                 ' Close the Workbook.
   xlApp.Quit               ' Close Microsoft Excel with the Quit method.
   
   ' Release the objects.
   
   Set xlSheet = Nothing
   Set xlBook = Nothing
   Set xlApp = Nothing
   
End Sub

Public Sub ExcelUpdate(InputWaveData() As Single, OutputWaveData() As Single, _
                       OutputWave As Wave, frm As Form, ByVal NumberOfSamples As Integer, _
                       ByVal SamplingFrequency As Double, ByVal Reset As Boolean, _
                       ByVal Save As Boolean)

    'Takes several parameters.  This function procedurally goes through and performs
    'all those tasks (or runs the routines if needed) that are needed to update the
    'Excel Worksheet.
    
    'VARIABLE DECLARATIONS
    Dim i As Integer, FourierInput As Integer
    Dim xlApp As Excel.Application
    Dim xlBook As Excel.Workbook
    Dim xlSheet As Excel.Worksheet
    Dim xlSheet2 As Excel.Worksheet
    Dim xlChart As Excel.Chart
    Dim xlFourierAmplitudeChart As Excel.Chart
    
    'VARIABLE INITIALIZATIONS
    'FourierInput is the number of samples we feed into excel's FFT function
    'It must be a multiple of two.  But since we don't want to cut any data,
    'we want the next largest integer power of 2 (i.e., don't let CInt() round
    'down).
    Select Case Log(NumberOfSamples) / Log(2) - CInt(Log(NumberOfSamples) / Log(2))
        Case Is <= 0 'In this case, we have a fractional part of >= .5 (since cint() rounds)
            FourierInput = 2 ^ CInt(Log(NumberOfSamples) / Log(2))  'Round down
        Case Is > 0 'In this case, we have a fractional part of <.5
            FourierInput = 2 ^ (CInt(Log(NumberOfSamples) / Log(2)) + 1)    'Round up
    End Select
            
    frm.StatusLabel.Caption = "Opening Excel. . ." 'takes time to set variables, so
                                                    'write this now.
    Set xlApp = New Excel.Application
    Set xlBook = xlApp.Workbooks.Open(frm.FileBoxExcelLink.Text)
    Set xlSheet = xlBook.Worksheets("Sheet1")
    Set xlSheet2 = xlBook.Worksheets("Sheet2")
    Set xlChart = xlSheet.ChartObjects(1).Chart
    Set xlFourierAmplitudeChart = xlSheet.ChartObjects(2).Chart

    'PROCEDURE
    If frm.chkMakeExcelVisible.value Then
        xlApp.Visible = True
        xlSheet.Visible = xlSheetVisible
    End If

    xlSheet.Activate
    
    frm.StatusLabel.Caption = "Updating Data. . ."
    
    'First, we have to make sure we have the FFT routines loaded.
    'If we don't load them now, then the form_load routines which they
    'contain won't be done in time to perform the FFT, and the update won't work.
    'Thus, we need to do it first thing.
    
    Call AddAddIns(xlApp)
    
    'Prevent Excel from slowing to a crawl by disabling auto-calculation.
    xlApp.Calculation = xlCalculationManual
    
    'Take values in array and write them in appropriate columns.
    Call WriteDataToCells(xlSheet, InputWaveData(), OutputWaveData(), OutputWave, _
                          FourierInput, NumberOfSamples, SamplingFrequency, _
                          frm.chkHanning.value)
    
    frm.StatusLabel.Caption = "Performing FFT. . ."
    
    'Recalculate just so it looks nice before FFT.
    xlApp.Calculate
    
    'Perform FFT and place Data in appropriate columns.
    Call FFT(xlApp, xlSheet, FourierInput, frm.chkHanning.value)
    
    frm.StatusLabel.Caption = "Recalculating. . ."
    xlApp.Calculate
    
    'Put Calculation back to where it was.
    xlApp.Calculation = xlCalculationAutomatic
    
    'Need to reset the charts to current values. . .
    Call ChartSetup(xlChart, xlSheet, NumberOfSamples, 0)
    Call ChartSetup(xlFourierAmplitudeChart, xlSheet, NumberOfSamples, 1)
    Call RemoveAddIns(xlApp)
    
    'Add data point to Bode Plot if option is selected.
    If Reset Then Call ClearFrequency(xlSheet2)
    If Save Then Call FrequencyUpdate(xlApp, xlBook, xlSheet, xlSheet2, OutputWave)
    xlSheet.Activate
    
    xlSheet.Range("A1").Select
    
    If frm.chkSaveAfterUpdate.value Then
        xlBook.Save             'Autosave it
    End If
    
    xlBook.Close
    xlApp.Quit

    frm.StatusLabel.Caption = "Updating window. . ."
    
    Set xlBook = Nothing
    Set xlSheet = Nothing
    Set xlChart = Nothing
    Set xlApp = Nothing
    
    frm.StatusLabel.Caption = "Stopped."
End Sub
Private Sub ChartSetup(xlChart As Excel.Chart, xlSheet As Excel.Worksheet, NumberOfSamples As Integer, WhichChart As Integer)
   'This routine sets all the variables on the charts which may need to be redefined
   'From acquisition to acquisition
   
   Select Case WhichChart
     Case 0 'This is our Stimulus / Response graph.
        '(Re)Set all of the chart's characteristics which we care about.
        xlChart.SeriesCollection(1).XValues = xlSheet.Range("$A$2:$A$" & (NumberOfSamples + 1))
        xlChart.SeriesCollection(1).Values = xlSheet.Range("$C$2:$C$" & (NumberOfSamples + 1))
        xlChart.SeriesCollection(2).XValues = xlSheet.Range("$A$2:$A$" & (NumberOfSamples + 1))
        xlChart.SeriesCollection(2).Values = xlSheet.Range("$D$2:$D$" & (NumberOfSamples + 1))
        
        With xlChart.Axes(xlCategory)
            .MinimumScaleIsAuto = False
            .MaximumScaleIsAuto = False
            .MinorUnitIsAuto = False
            .MajorUnitIsAuto = False
            .Crosses = xlCustom
            .ReversePlotOrder = False
            .ScaleType = xlLinear
            .MinimumScale = 0
            .MaximumScale = NumberOfSamples
            .MinorUnit = NumberOfSamples \ 20
            .MajorUnit = NumberOfSamples \ 4
            .CrossesAt = 0
            .ReversePlotOrder = False
            .ScaleType = xlLinear
        End With
      Case 1    'FFT chart
        xlChart.SeriesCollection(1).XValues = xlSheet.Range("$B$2:$B$" & (NumberOfSamples + 1))
        xlChart.SeriesCollection(1).Values = xlSheet.Range("$G$2:$G$" & (NumberOfSamples + 1))
        xlChart.SeriesCollection(2).XValues = xlSheet.Range("$B$2:$B$" & (NumberOfSamples + 1))
        xlChart.SeriesCollection(2).Values = xlSheet.Range("$H$2:$H$" & (NumberOfSamples + 1))
   
        With xlChart.Axes(xlCategory)
            .MinimumScaleIsAuto = True
            .MaximumScaleIsAuto = True
            .MinorUnitIsAuto = False
            .MajorUnitIsAuto = False
            .Crosses = xlCustom
            .ReversePlotOrder = False
            .ScaleType = xlLogarithmic
            .MinorUnit = 10
            .MajorUnit = 10
            .ReversePlotOrder = False
        End With
        xlChart.Axes(xlCategory).CrossesAt = xlChart.Axes(xlCategory).MinimumScale
        
        With xlChart.Axes(xlValue)
            .MinimumScaleIsAuto = True
            .MaximumScaleIsAuto = True
            .MinorUnitIsAuto = True
            .MajorUnitIsAuto = False
            .MajorUnit = 20
            .Crosses = xlCustom
            .CrossesAt = xlCustom
            .ReversePlotOrder = False
            .ScaleType = xlLinear      'Since y-axis is dB, it's already log.
        End With
      
    End Select
End Sub
Private Sub AddAddIns(xlApp As Excel.Application)
    'Does exactly what the name says.
    'Sometimes Excel acts very strangely with regards to this.  Sometimes it will
    'report the add-ins as added even though they are not, or vice versa.  As long
    'as you let the application load and unload the add-ins, there should be
    'no problem.
    
    With xlApp
    
        If Not .AddIns("Analysis ToolPak").Installed Then
            .AddIns("Analysis ToolPak").Installed = True
        End If
    
        If Not .AddIns("Analysis ToolPak - VBA").Installed Then
            .AddIns("Analysis ToolPak - VBA").Installed = True
        End If
    
    End With
    
End Sub
Private Sub RemoveAddIns(xlApp As Excel.Application)
    'Does exactly what the name says.
    
    With xlApp
    
        If .AddIns("Analysis ToolPak").Installed Then
            .AddIns("Analysis ToolPak").Installed = False
        End If
        If .AddIns("Analysis Toolpak - VBA").Installed Then
            .AddIns("Analysis ToolPak - VBA").Installed = False
        End If
    
    End With
    
End Sub
Private Sub FFT(xlApp As Excel.Application, xlSheet As Excel.Worksheet, _
               ByVal FourierInput As Integer, HanningWindow As Boolean)
    'Runs the FFT algorithm as given by Excel Add-Ins.

    Dim i As Integer
    With xlApp
    
        If Not .AddIns("Analysis ToolPak").Installed Then Call AddAddIns(xlApp)
        If Not .AddIns("Analysis ToolPak - VBA").Installed Then Call AddAddIns(xlApp)
    
    End With
    
    'Just in case the Addins were just added, this loop will make sure that
    'the auto-initialize routine is complete before we actually try the FFT.
    
    With xlSheet
    
        .Range("$E$2:$F$" & (FourierInput + 1)).Clear
    
        'This loop is just to ensure that the Addins are really loaded and
        'ready to go.  ATPVBAEN.XLA has some initialization code it runs
        'when it loads, and it must be complete before performing the
        'FFT.  This test tries to make sure that the code is complete
        'by looping until commands are actually executed.
        
        Do Until IsEmpty(.Range("$E$2")) And IsEmpty(.Range("$F$" & (FourierInput + 1)))
            DoEvents
        Loop
    
    End With
    
    With xlApp
    
        If HanningWindow Then   'Use the Hanning'ed Data
            .Run "ATPVBAEN.XLA!Fourier", xlSheet.Range("$L$2:$L$" & (FourierInput + 1)), _
            xlSheet.Range("$E$2:$E$" & (FourierInput + 1)), False, False
            .Run "ATPVBAEN.XLA!Fourier", xlSheet.Range("$M$2:$M$" & (FourierInput + 1)), _
            xlSheet.Range("$F$2:$F$" & (FourierInput + 1)), False, False
        Else                    'Use the regular data.
            .Run "ATPVBAEN.XLA!Fourier", xlSheet.Range("$C$2:$C$" & (FourierInput + 1)), _
            xlSheet.Range("$E$2:$E$" & (FourierInput + 1)), False, False
            .Run "ATPVBAEN.XLA!Fourier", xlSheet.Range("$D$2:$D$" & (FourierInput + 1)), _
            xlSheet.Range("$F$2:$F$" & (FourierInput + 1)), False, False
        End If
    
    End With
    
    xlSheet.Range("$A$1").Select
    
End Sub
Private Sub WriteDataToCells(xlSheet As Excel.Worksheet, InputWaveData() As Single, _
            OutputWaveData() As Single, OutputWave As Wave, ByVal FourierInput As Integer, _
            ByVal NumberOfSamples As Integer, ByVal SamplingFrequency As Double, _
            HanningWindow As Boolean)
    Dim i As Integer
    
    With xlSheet
        
        'First, write the raw data to their respective columns.
        For i = 0 To NumberOfSamples - 1
            .Cells(i + 2, 1) = i + 1
            .Cells(i + 2, 3) = OutputWaveData(i)
            .Cells(i + 2, 4) = InputWaveData(i)
        Next i
        
        'Next, we rewrite and some other cells (in case someone does
        'something like overwrite a cell and screw up the works, this will
        'correct it).
        
        .Range("$B$2") = "=($A2*$N$19)/$N$20"
        .Range("$G$2") = "=IF(IMABS($E2)=0,-100,20*LOG10(IMABS($E2)))"
        .Range("$H$2") = "=IF(IMABS($F2)=0,-100,20*LOG10(IMABS($F2)))"
        .Range("$I$2") = "=DEGREES(ATAN2(IMREAL($E2),IMAGINARY($E2)))"
        .Range("$J$2") = "=DEGREES(ATAN2(IMREAL($F2),IMAGINARY($F2)))"
        .Range("$A$" & (FourierInput + 2) & ":$T$37768").Clear
        .Range("$B$2:$B$" & (FourierInput + 1)).FillDown
        .Range("$G$2:$J$" & (FourierInput + 1)).FillDown
        .Range("N2") = "=(MAX($C$2:$C$" & (NumberOfSamples + 1) & ")-MIN($C$2:$C$" & (NumberOfSamples + 1) & "))/2"
        .Range("N4") = "=(MAX($D$2:$D$" & (NumberOfSamples + 1) & ")-MIN($D$2:$D$" & (NumberOfSamples + 1) & "))/2"
        .Range("N6") = "=$N$4/$N$2"
        .Range("N8") = "=20*LOG10($N$6)"
        .Range("N10") = GetPhase(InputWaveData(), OutputWaveData(), OutputWave, NumberOfSamples)
        .Range("N12") = "=(1/$N$14)"
        .Range("N14") = OutputWave.Frequency
        .Range("N17") = OutputWave.Cycles
        .Range("N18") = FourierInput
        .Range("N19") = SamplingFrequency
        .Range("N20") = NumberOfSamples
        
        'Zero out the balance of samples between NumberOfSamples and the
        'next whole power of 2 (i.e., pad with 0's).  Remember that
        'FourierInput should represent the next highest power of 2.
        If FourierInput > NumberOfSamples Then
            For i = NumberOfSamples To FourierInput
                .Cells(i + 2, 1) = i + 1
                .Cells(i + 2, 3) = 0
                .Cells(i + 2, 4) = 0
            Next i
        End If
    
        'If the Hanning Window check box is checked, then do those columns.
        If HanningWindow Then
            .Range("$K$2") = "=(0.5-0.5*COS(2*PI()*$A2/$N$18))"
            .Range("$K$2:$K$" & (FourierInput + 1)).FillDown
            .Range("$L$2") = "=$C2*$K2"
            .Range("$M$2") = "=$D2*$K2"
            .Range("$L$2:$L$" & (FourierInput + 1)).FillDown
            .Range("$M$2:$M$" & (FourierInput + 1)).FillDown
            
            If FourierInput > NumberOfSamples Then
                For i = NumberOfSamples To FourierInput
                    .Cells(i + 2, 22) = 0
                    .Cells(i + 2, 23) = 0
                Next i
            End If
        Else    'If not, clear them.
            .Range("$K$2:$M$" & (FourierInput + 1)).Clear
        End If
        
        
    End With
    
End Sub

Private Sub FrequencyUpdate(xlApp As Excel.Application, xlBook As Excel.Workbook, xlSheet As Excel.Worksheet, _
                            xlSheet2 As Excel.Worksheet, OutputWave As Wave)
    'This function is called to update Sheet2 of the Excel worksheet.  It puts calculated values of
    'Gain (dB and magnitude) and Phase into the worksheet, and then updates the graph characteristics.
    
    Dim GainPlot As Excel.Chart, PhasePlot As Excel.Chart
    Dim i As Integer, Entries As Integer, OverWrite As Integer
    Const PI As Double = 3.141592653589

    Set GainPlot = xlSheet2.ChartObjects(1).Chart
    Set PhasePlot = xlSheet2.ChartObjects(2).Chart
    
    With xlSheet2
        'Important we activate the sheet before anything else.
        .Activate
    
        'Look through the current list of data, to see if there is a frequency value which
        'matches the one we have right now.  If so, store it in Overwrite.
        
        i = 2
        Do While (xlSheet2.Cells(i, 1) <> "")
            If xlSheet2.Cells(i, 1) = OutputWave.Frequency Then
                OverWrite = i
            End If
            i = i + 1
        Loop

        Entries = i - 2
        
        If OverWrite = 0 Then   'New frequency entry.
            .Cells(i, 1) = OutputWave.Frequency
            .Cells(i, 2) = 2# * PI * OutputWave.Frequency
            .Cells(i, 3) = xlSheet.Range("N6")
            .Cells(i, 4) = "=20*log10($C" & i & ")"
            .Cells(i, 5) = xlSheet.Range("N10")
        Else    'Overwrite old frequency entry.
            .Cells(OverWrite, 1) = OutputWave.Frequency
            .Cells(OverWrite, 2) = 2# * PI * OutputWave.Frequency
            .Cells(OverWrite, 3) = xlSheet.Range("N6")
            .Cells(OverWrite, 4) = "=20*log10($C" & OverWrite & ")"
            .Cells(OverWrite, 5) = xlSheet.Range("N10")
        End If
    
        If Entries > 1 Then     '(Re)sort the data.
            .Range("$A$2:$E$" & (Entries + 2)).Select
            xlApp.Selection.Sort Key1:=.Range("$A$2:$A$" & (Entries + 2)), Order1:=xlAscending, _
                    Header:=xlNo, OrderCustom:=1, MatchCase:=False, Orientation:=xlTopToBottom
            .Range("A1").Select
            
            GainPlot.SeriesCollection(1).XValues = xlSheet2.Range("$A$2:$A$" & (Entries + 2))
            GainPlot.SeriesCollection(1).Values = xlSheet2.Range("$D$2:$D$" & (Entries + 2))
            PhasePlot.SeriesCollection(1).XValues = xlSheet2.Range("$A$2:$A$" & (Entries + 2))
            PhasePlot.SeriesCollection(1).Values = xlSheet2.Range("$E$2:$E$" & (Entries + 2))
        End If
    
    End With
        
    'Set all of the values and characteristics relating to this graph.
    With GainPlot.Axes(xlCategory)
        .MinimumScaleIsAuto = True
        .MaximumScaleIsAuto = True
        .MinorUnitIsAuto = False
        .MajorUnitIsAuto = False
        .Crosses = xlCustom
        .CrossesAt = .MinimumScale
        .ReversePlotOrder = False
        .MajorUnit = 10
        .MinorUnit = 10
        .ReversePlotOrder = False
        .ScaleType = xlLogarithmic
    End With
    With GainPlot.Axes(xlValue)
        .MinimumScaleIsAuto = True
        .MaximumScaleIsAuto = True
        .MinorUnitIsAuto = True
        .MajorUnitIsAuto = True
        .Crosses = xlCustom
        .CrossesAt = .MinimumScale
        .ReversePlotOrder = False
        '.MajorUnit = 20     'dB    --But for small ranges of dB, this stinks--leave it auto.
        .CrossesAt = 0
        .ReversePlotOrder = False
        .ScaleType = xlLinear
    End With
    
    With PhasePlot.Axes(xlCategory)
        .MinimumScaleIsAuto = True
        .MaximumScaleIsAuto = True
        .MinorUnitIsAuto = False
        .MajorUnitIsAuto = False
        .Crosses = xlCustom
        .CrossesAt = .MinimumScale
        .ReversePlotOrder = False
        .MajorUnit = 10
        .MinorUnit = 10
        .ReversePlotOrder = False
        .ScaleType = xlLogarithmic
    End With
    With PhasePlot.Axes(xlValue)
        .MinimumScaleIsAuto = True
        .MaximumScaleIsAuto = True
        .MinorUnitIsAuto = True
        .MajorUnitIsAuto = False
        .Crosses = xlCustom
        .ReversePlotOrder = False
        .MajorUnit = 45     'deg
        .CrossesAt = 0
        .ReversePlotOrder = False
        .ScaleType = xlLinear
        If .MinimumScale < -180 Then
            .MinimumScaleIsAuto = False
            .MinimumScale = -180
        End If
        
        If .MaximumScale > 180 Then
            .MaximumScaleIsAuto = False
            .MaximumScale = 180
        End If
    End With
      
    GainPlot.Axes(xlValue).CrossesAt = GainPlot.Axes(xlValue).MinimumScale
    PhasePlot.Axes(xlValue).CrossesAt = PhasePlot.Axes(xlValue).MinimumScale

End Sub

Private Sub ClearFrequency(xlSheet2 As Excel.Worksheet)
    'This function deletes all the gathered frequency data saved in the spreadsheet.

    Dim i As Integer, Entries As Integer
    xlSheet2.Activate
    
    'First, gather number of entries.
    i = 2
    Do While (xlSheet2.Cells(i, 1) <> "")
        i = i + 1
    Loop

    Entries = i - 2

    'Delete
    xlSheet2.Range("$A$2:$E$" & i).Clear
End Sub
